/*!
 * \file    xm-type.h
 *
 * \brief   Spezifische Datentypen fr externe Messgerte
 *
 *          Die Datentypen, die in dieser Datei deklariert werden,
 *          sind spezifisch fr den Zweck "Messtechnik"
 *
 * \version 2.1
 * \date    20.09.2017
 * \author  Dr. S. Fischer, gedas deutschland GmbH (sfi)
 * \author  Dr. M. Tschersich, T-Systems Enterprise Services GmbH (mts)
 *
 * \hist    \li sfi 15.08.2005   Aufzhlungstypen fr Messgren und
 *                               Gerteeigenschaften hinzugefgt.
 *          \li sfi 17.08.2005   Typ TTChannelFeature hinzugefgt, um dynamische
 *                               Eigenschaften abzubilden (z.B. Frequenzgang)
 *          \li sfi 17.08.2005   TTPhysicalQuantity um den Wert 'undefined' erweitert,
 *                               um Vorgabewerte benutzen zu knnen
 *                               (vgl. createparaminitialquantity).
 *          \li sfi 18.08.2005   Datentypen fr zeitaufgelste Messungen hinzugefgt
 *          \li mts 27.10.2005   Review und grundlegende berarbeitung.
 *          \li mts 30.07.2007   TeFilter entfernt und durch TuFilter ersetzt;
 *                               Aufzhlungstyp fr Ergebnis Diodenmessung hinzugefgt
 *                               und in union TuResultValue integriert; Sequence fr
 *                               Messwerte bzgl. Verwendung von TuResultValue (statt
 *                               TValue) umgestellt; Bugfix: falschen Identifier 
 *                               TeCalibrationStatus durch TeCalibrationState ersetzt (mts).
 *          \li mts 19.10.2007   unterschiedliche Aufzhlungstypen fr Strom inline
 *                               (eCurrentInline) und ber Zangenamperemeter gemessen
 *                               (eCurrentPliers) zu einem einzigen Typ (eCurrent)
 *                               zusammengefasst; Datentyp TSensorId eingefgt; weiteren
 *                               Zustand eCplGND bei Kopplung eingefhrt; zustzliche
 *                               Datentypen TLogicalConnectorSetup und TsqLogicalConnectorSetup
 *                               eingefhrt zwecks Differenzierung bei Setup und Eigenschaften-
 *                               Abfrage logischer Anschlsse; weiteren Aufzhlungstyp eNotReady
 *                               beim Aktivierungsstatus (TeActivationStatus) eingefgt; Struktur
 *                               TMeasurementCycle um Min/Max-Werte fr Zeitstempel und um Angabe
 *                               der zum Triggerzeitpunkt verwendeten Triggerschwelle erweitert;
 *                               Klarstellung zum Umgang mit vom Gert nicht einstellbaren
 *                               (i.e. zu hohen) Sampleraten bei TSampleConfig ergnzt
 *          \li mts 07.12.2007   TMeasurementCycle um neues Struct-Member ulSequenceNumber ergnzt
 *          \li mts 25.07.2008   neuen Datentyp fr Vorverdichtungsmethode (TeSampleMethod) eingefhrt;
 *                               Struktur "TSampleConfig" um Member fr Vorverdichtungsmethode
 *                               (sampleMethod) erweitert
 *          \li rde 20.09.2017   neue Typen TVersion, TPluggedSensor und TsqPluggedSensor fr 
 *                               neue Attribute Device::seqPluggedSensors und Device::firmwareVersion
 */


/// Bezeichnung einer Buchse.
typedef  A_ASCIISTRING_32        TSocketName;


/// Liste von Buchsennamen.
typedef  sequence<TSocketName>   TsqSocketNames;


/// Name eines logischen Anschlusses.
typedef  A_ASCIISTRING_32        TLogicalName;


/// Kanalname.
typedef  A_ASCIISTRING_32        TChannelName;


/// Sensor ID.
typedef  A_ASCIISTRING_32        TSensorId;


/// Messwert.
typedef  A_FLOAT64               TValue;

/*!
 * \brief   Sequenz mehrerer Messwerte.
 */
typedef sequence<TValue>   TsqValues;


/// Arten zur Verwendung optionaler Werte.
typedef enum TeOption
{
   eOptUse,    ///< Der Wert soll verwendet werden (ist gltig).
   eOptIgnore  ///< Der Wert soll ignoriert werden (ist ungltig).
   
} TeOption;


/*!
 * \brief   Optionaler Wert.
 */
typedef struct TOptionalValue
{
   TeOption    eOption;    ///< Angabe zur Gltigkeit des Wertes.
   TValue      value;      ///< Der Wert selbst (nur zu verwenden, wenn eOption auf eOptUse gesetzt ist).
   
} TOptionalValue;

/*!
 * \brief   Sequenz mehrerer optionaler Werte.
 */
typedef sequence<TOptionalValue> TsqOptionalValues;


/*!
 * 
 */


/*!
 * \brief   Koordinatenangabe fr die Lage einer Anschlussbuchse auf der Frontplatte.
 */
typedef struct TSocketCoord
{
   A_UINT16    xpos;    ///< horizontale Entfernung vom linken Rand in Teilen von Tausend.
   A_UINT16    ypos;    ///< vertikale Entfernung vom oberen Rand in der gleichen Skalierung wie das horizontale Ma.
   
} TSocketCoord;

/*!
 * \brief   Datentyp zur Beschreibung einer Anschlussbuchse am Messgert.
 */
typedef struct TSocket
{
   TSocketName       name;    ///< Name der Anschlussbuchse so wie am Gert (i.d.R. Frontplatte) bzw. in der Anleitung bezeichnet.
   TSocketCoord      position;///< Position der Anschlussbuchse auf der Frontplatte des Gertes.
   
} TSocket;

/*!
 * \brief   Kombination mehrerer Anschlussbuchsen (zur Beschreibung eines \ref TLogicalConnector "logischen Anschlusses").
 */
typedef sequence<TSocket> TsqSockets;


/*!
 * \brief   Vollstndige Information zu einem logischen Anschluss.
 */
typedef struct TLogicalConnector
{
   TLogicalName      name;    ///< Eindeutiger Name des logischen Anschlusses.
   TsqSockets        sockets; ///< Die Buchsenkombination, deren Gesamtheit (i.d.R. eine oder zwei Buchsen) den logischen Anschluss darstellt.
  
} TLogicalConnector;


/*!
 * \brief   Gesamtheit aller verfgbaren logischen Anschlsse eines Gertes.
 */
typedef sequence<TLogicalConnector> TsqLogicalConnectors;


/*!
 * \brief   Konfiguration zum Definieren von logischen Anschlssen.
 */
typedef struct TLogicalConnectorSetup
{
   TLogicalName      name;          ///< Eindeutiger zu verwendender Name des logischen Anschlusses.
   TsqSocketNames    socketNames;   ///< Die Namen der Buchsen, deren Gesamtheit (i.d.R. eine oder zwei Buchsen) den logischen Anschluss definiert.
   
} TLogicalConnectorSetup;


/*!
 * \brief   Konfiguration aller verfgbaren logischen Anschlsse eines Gertes.
 */
typedef sequence<TLogicalConnectorSetup> TsqLogicalConnectorSetup;


/*!
 * \brief   An einer Buchse angesteckter Sensor.
 */
typedef struct TPluggedSensor
{
   TSocketName       socketName; ///< Name des physikalischen Anschlusses (Buchse).
   TSensorId         sensorId;   ///< Identifier des angesteckten Sensors.
   
} TPluggedSensor;


/*!
 * \brief   Die an logischen Anschlssen angesteckten Sensoren.
 */
typedef sequence<TPluggedSensor> TsqPluggedSensors;


/*!
 * \brief   Beschreibung einer unspezifischen Gerteeigenschaft.
 */
typedef struct TProperty
{
   A_ASCIISTRING_32  key;     ///< Schlsselwort.
   A_ASCIISTRING_32  value;   ///< Wert als Text.
   
} TProperty;


/*!
 * \brief   Liste (Sequenz) von unspezifischen Gerteeigenschaften.
 */
typedef  sequence<TProperty>  TsqProperties;


/*!
 * \brief   Mgliche physikalische Messgren.
 */
typedef enum TePhysicalQuantity
{
  eVoltage,          ///< Spannung.
  eCurrent,          ///< Strom (sowohl inline als auch ber Zangenamperemeter gemessen).
  eTemperature,      ///< Temperatur.
  eResistanceOhmic,  ///< Ohmscher Widerstand.
  ePressure,         ///< Druck.
  eDiodeCheck        ///< Diodenprfung.
  
} TePhysicalQuantity;

/*!
 * \brief   Mgliche Ergebnisse einer Diodenprfung.
 */
typedef enum TeDiodeResult
{
   eForwardBias,  ///< Durchlassrichtung, d.h. die technische Stromrichtung verluft von der Anode zur Kathode.
   eReverseBias,  ///< Sperrrichtung, d.h. die technische Stromrichtung verluft von der Kathode zur Anode.
   eShortCircuit, ///< Kurzschluss (Diode in beiden Richtungen leitend).
   eOpenCircuit   ///< Unterbrechung (Diode in beiden Richtungen nichtleitend).
   
} TeDiodeResult;


/*!
 * \brief   Sequenz von Ergebnissen einer Diodenprfung.
 */
 typedef sequence<TeDiodeResult> TsqDiodeResult;


/*!
 * \brief   Mgliche Kopplungen eines Kanals.
 */
typedef enum TeCoupling
{
   eCplNotApplic, ///< Nicht zutreffend, d.h. der Begriff \em Kopplung ergibt bei der aktuell eingestellten Gre keinen Sinn.
   eCplAC,        ///< Kopplung gegen Wechselstrom (Alternating Current).
   eCplDC,        ///< Kopplung gegen Gleichstrom (Direct Current).
   eCplGND        ///< Kopplung gegen Masse (Ground).

} TeCoupling;


/*!
 * \brief   Die verschiedenen Modi fr die Messbereichseinstellung.
 */
typedef enum TeRangeMode
{
   eAutoRange,       ///< Gert stellt den Messbereich automatisch ein.
   eManualRange      ///< Der Anwender gibt einen Messbereich vor.
   
} TeRangeMode;


/*!
 * \brief   Darstellung einer Messbereichsauswahl.
 */
typedef struct TRange
{
   TeRangeMode eRangeMode;    ///< Die Art der Messbereichsauswahl (automatisch/ manuell).
   TValue      lowerLimit;    ///< Die untere Grenze des Messbereichs (nur relevant bei \ref eManualRange "manueller" Auswahl).
   TValue      upperLimit;    ///< Die obere Grenze des Messbereichs (nur relevant bei \ref eManualRange "manueller" Auswahl).
   A_BOOLEAN   isOverridable; ///< Gibt an, ob der Messbereich durch den bei einer Messung angegebenen erwarteten Wert modifiziert werden darf (nur relevant bei \ref eManualRange "manueller" Auswahl).
   
} TRange;


/*!
 * \brief   Mgliche Kalibrierzustnde des Gertes.
 */
typedef enum TeCalibrationState
{
  eCalOk,            ///< Gert ist kalibriert.
  eCalIncomplete,    ///< weitere Kablibrierschritte erforderlich.
  eCalRequired,      ///< Kalibrierung ist erforderlich.
  eCalBusy           ///< Ein Kalibrierschritt luft gerade.
  
} TeCalibrationState;


/*!
 * \brief   Mgliche Zustnde des Gertes.
 */
typedef enum TeDeviceState
{
   eBasicState,            ///< Grundzustand.
   eReadyForMeasurement,   ///< Messbereit.
   eCommError              ///< Kommunikationsstrung.
   
} TeDeviceState;


/*!
 * \brief   Mgliche Triggerflanken.
 */
typedef enum TeTriggerFlank
{
   eFlankRising,     ///< Trigger lst auf der ansteigenden Flanke aus.
   eFlankFalling     ///< Trigger lst auf der abfallenden Flanke aus.
   
} TeTriggerFlank;


/*!
 * \brief   Triggerschwelle.
 */
typedef struct TTriggerLevel
{
   /*!
    * \brief   Angabe, ob die Triggerschwelle automatisch ermittelt werden soll.
    *
    *          Wird dieser Wert auf TRUE gesetzt, wird die Triggerschwelle automatisch
    *          ermittelt und anschlieend dynamisch nachgefhrt. In diesem Fall wird
    *          die in der Struktur \ref TTrigger eingestellte \ref TTrigger::triggerTime "Triggerzeit"
    *          ignoriert.
    */
   A_BOOLEAN      autoSetup;
   TValue         levelValue;    ///< Wert der Triggerschwelle.
   TeTriggerFlank eTriggerFlank; ///< Art der Flanke, die zur Auslsung des Triggers fhren soll.
   A_BOOLEAN      dynaLevel;     ///< Angabe, ob die Triggerschwelle bei sich ndernder Signalamplitude dynamisch nachgefhrt werden soll.

} TTriggerLevel;


/*!
 * \brief   Beschreibung eines Triggerereignisses.
 *
 *          Mithilfe dieser Datenstruktur wird ein Ereignis beschrieben,
 *          das den Beginn einer neuen Messwerte-Sequenz beschreibt.
 *
 * \image latex Triggerverzoegerung.png "Beziehung zwischen Triggerereignis, -verzgerung und Aufzeichnungsintervall" width=12cm
 * \image html  Triggerverzoegerung.png "Beziehung zwischen Triggerereignis, -verzgerung und Aufzeichnungsintervall"
 *
 */
typedef struct TTrigger
{
   TChannelName      inputChannel;  ///< Der Name des Kanals, dessen Werte als Triggereingang verwendet werden sollen.
   TTriggerLevel     triggerLevel;  ///< Einstellungen fr die Festlegung der Triggerschwelle.
   TValue            delay;         ///< Zeitverschiebung (in Sekunden) zwischen Triggerereignis und Aufzeichnungsbeginn der Daten.
   TValue            recInterval;   ///< Lnge des Aufzeichnungsintervalls (in Sekunden). Ist dieser Wert gleich Null, luft die Aufzeichnung beliebig lange, jedoch maximal bis zum Eintreten des nchsten Trigger-Ereignisses.
   /*!
    * \brief   Zeitintervall bis zum erwarteten Auftreten des Triggerereignisses.
    *
    *          Ist dieser Wert gleich Null, wird beliebig lange auf ein Trigger-Ereignis gewartet.
    *          Falls der bei der \ref TTrigger::triggerLevel "Triggerschwelle" angegebene Wert fr
    *          \ref TTriggerLevel::autoSetup "autoSetup" auf TRUE gesetzt ist, wird das hier 
    *          angegebene Zeitintervall ignoriert.
    */
   TValue            triggerTime;   

} TTrigger;


/*!
 * \brief   Mgliche Arten fr die Auslsung der bertragung von Messwerten.
 */
 typedef enum TeTransmissionMode
{
   eTransByBlocksize,      ///< Die Daten werden bei Erreichen einer bestimmten Anzahl (Blockgre) von Messwerten bertragen.
   eTransByTime,           ///< Die Datenbertragung erfolgt, nachdem eine definierte Zeit lang Messwerte erfasst worden sind.
   eTransOnTrigger         ///< Die bertragung der Daten erfolgt, nachdem ein Triggerereignis vollstndig abgearbeitet ist.
   
} TeTransmissionMode;


/*!
 * \brief   Beschreibung der bertragungskonfiguration.
 */
typedef union TuTransmissionConfig switch (TeTransmissionMode)
{
   case eTransByBlocksize: A_UINT16    blocksize;           ///< Blockgre (Anzahl aufgezeichneter Messwerte), die eine Datenbertragung auslst.
   case eTransByTime:      TValue      transmissionRate;    ///< Anzahl geforderter datenbertragungen pro zeiteinheit (Sekunde).
   case eTransOnTrigger:   TTrigger    trigger;             ///< Beschreibung eines Triggerereignisses, dessen vollstndige Abarbeitung die Datenbertragung auslst.

} TuTransmissionConfig;


/*!
 * \brief   Aufzhlung der am Gert einstellbaren Vorverdichtungsmethoden.
 *
 *          In Abhngigkeit von den physikalischen Gegebenheiten des Gertes bzw. seines
 *          Signalprozessors entscheidet dieses selbstndig unter Bercksichtigung des
 *          Abtasttheorems, ob \em jeder erfasste Wert bertragen wird oder ob 
 *          eine Mittelung durchgefhrt wird. Die Einstellung eNormal beinhaltet ausdrcklich
 *          beide dieser Mglichkeiten.
 */
typedef enum TeSampleMethod
{
   eNormal,    ///< Erfassung von Einzelmesspunkten oder Mittelung ber jeweils mehrere erfasste Einzelmesspunkte (ber die verwendete Anzahl von Messpunkten entscheidet das Gert autark).
   eMinOnly,   ///< Der kleinste in einer bestimmten Anzahl von Einzelmesspunkten vorkommende Wert (ber die verwendete Anzahl von Messpunkten entscheidet das Gert autark).
   eMaxOnly,   ///< Der grte in einer bestimmten Anzahl von Einzelmesspunkten vorkommende Wert (ber die verwendete Anzahl von Messpunkten entscheidet das Gert autark).
   eMinMax     ///< Der kleinste und der grte in einer bestimmten Anzahl von Einzelmesspunkten vorkommende Wert (ber die verwendete Anzahl von Messpunkten entscheidet das Gert autark).
   
} TeSampleMethod;


/*!
 * \brief   Konfiguration fr eine kontinuierliche Datenbertragung.
 */
typedef struct TSampleConfig
{
   /*!
    * \brief   Die von Anwender fr die Messung gewnschte Abtastrate.
    *
    *          Die Anzahl der pro Sekunde zu erfassenden Messwerte. Der hier 
    *          Wert stellt zunchst nur den Anwenderwunsch dar. Dieser muss nicht
    *          zwingend durch die i.d.R. diskreten mglichen Abtastraten des Gertes
    *          abgedeckt sein. Daher wird vom Gert in dem Fall, dass keine dem
    *          Anwenderwunsch entsprechende Abtastrate zur Verfgung steht, die
    *          nchsthhere mgliche Abtastrate eingestellt. Der tatschlich eingestellte
    *          Wert wird per GDI_InformationReport an das Attribut
    *          \ref xm::Channel::currentSampleRate "currentSampleRate" des Kanals
    *          mitgeteilt.\n Falls der
    *          angegebene Wert die physikalischen Fhigkeiten des Gertes
    *          bersteigt, muss das Gert die ihm maximal mgliche Abtastrate
    *          einstellen. Der Treiber muss der Anwendung in diesem Fall eine
    *          Warnung senden, dass die angeforderte Abtastrate vom Gert nicht
    *          untersttzt und statt dessen ein anderer Wert (dieser ist
    *          in dem Warnungstext zu nennen) verwendet wird.
    */
   TValue      sampleRate;
   
   
   /*!
    * \brief   Die zu verwendende Vorverdichtungsmethode.
    *
    * \note    Bestimmte Vorverdichtungsmethoden knnen zu der gewhlten Abtastrate
    *          inkompatibel sein (z.B. wenn berhaupt keine Verdichtung durchgefhrt
    *          werden soll und daher in diesem Modus die maximal mgliche Abtastrate
    *          des Gertes eingeschrnkt ist). In diesem Fall kommt das bei dem
    *          Parameter sampleRate beschriebene Verfahren ebenfalls zum Einsatz.
    */
   TeSampleMethod sampleMethod;
   
   /*!
    * \brief   bertragungskonfiguration.
    *
    *          Das Kriterium zum Auslsen einer bertragung der aufgezeichneten Werte
    *          an die Anwendung.
    */
   TuTransmissionConfig transConfig;
   
} TSampleConfig;


/*!
 * \brief   Zeitstempel.
 */
typedef TValue TTimestamp;


/*!
 * \brief   Wert mit Zeitstempel versehen.
 */
typedef struct TTimedValue
{
   TValue      resultValue;      ///< Messwert.
   /*!
    * \brief   Zeitstempel.
    *
    *          Der Nullpunkt (t=0) liegt bei \ref eTransOnTrigger "getriggerter Messung" immer
    *          beim Triggerzeitpunkt und wird auch fr jeden neuen Trigger-Zyklus wieder 
    *          bei Null begonnen (damit haben die bei negativer \ref TTrigger::delay "Verzgerung"
    *          \em vor dem Triggerzeitpunkt liegenden Messwerte einen \em negativen
    *          Zeitstempel. Bei ungetriggerter Messung wird die Zeit ab Beginn der Messung 
    *          fortlaufend weitergezhlt (auch im nchsten bertragenen Datenblock!).
    */
   TTimestamp  timestamp;

} TTimedValue;


/*!
 * \brief   Sequenz von mit Zeitstempel versehenen Messwerten.
 */
typedef sequence<TTimedValue> TsqTimeValues;


/*!
 * \brief   Struktur aus einer Sequenz von Messwerten, erweitert um Informationen zu
 *          deren Extrema (y-Werte) und der verwendeten Vorverdichtungsmethode.
 *
 * \note    Die erweiterten Informationen sind nur dann mit sinnvollen Werten zu belegen,
 *          wenn die Sequenz \em values mindestens ein Element enthlt. Wenn die Sequenz
 *          leer ist, bedeutet dies, dass die gesamte Struktur nicht verwendet wird.
 */
typedef struct TExtendedValueSequence
{
   TsqValues         values;        ///< Sequenz mit den Messwerten (zeitlich quidistant).
   TTimedValue       minValue;      ///< Kleinster in der Sequenz enthaltener Wert.
   TTimedValue       maxValue;      ///< Grter in der Sequenz enthaltener Wert.
   TeSampleMethod    sampleMethod;  ///< Die Vorverdichtungsmethode, die zur Entstehung der Messwerte gefhrt hat.
   
} TExtendedValueSequence;


/*!
 * \brief   Messwertezyklus.
 *
 *          Diese Struktur beinhaltet neben der Sequenz mit den
 *          aufgezeichneten Werten auch das 
 *          darin enthaltene Minimum und Maximum.
 */
typedef struct TMeasurementCycle
{
   /*!
    * \brief   Erste Sequenz mit den Messwerten und erweiterten Informationen.
    *
    * \note    Die Verwendung dieser Messwertesequenz ist mandatorisch, d.h. die darin
    *          enthaltene \ref TExtendedValueSequence::values "Sequenz von Messwerten" muss
    *          mit mindestens einem Wert gefllt sein. Einzige Ausnahme hiervon ist die
    *          Diodenmessung; bei dieser ist nur die speziell fr
    *          \ref TMeasurementCycle::diodeSequence "Diodenmesswerte bestimmte Sequenz" gefllt.
    */
   TExtendedValueSequence  valueSequence1;

   /*!
    * \brief   Zweite Sequenz mit den Messwerten und erweiterten Informationen.
    *
    * \note    Die Verwendung dieser Messwertesequenz ist optional. Falls diese verwendet wird,
    *          muss sich Verdichtungsmethode von der in der ersten Sequenz unterscheiden. Bei
    *          einer Messung zur Diodenprfung darf diese Sequenz \em keinesfalls gefllt sein.
    */
   TExtendedValueSequence  valueSequence2;
   
   /*!
    * \brief   Sequenz von Diodenmesswerten.
    *
    * \note    Diese Sequenz darf nur dann eine von Null verschiedene Anzahl von Elementen besitzen,
    *          wenn der betreffende Kanal fr eine \ref TePhysicalQuantity::eDiodeCheck "Diodenprfung"
    *          verwendet wird; ansonsten \em muss die Sequenz leer sein.
    */
   TsqDiodeResult          diodeSequence;

   TTimestamp              minTime;          ///< Kleinster in jeder der Sequenzen enthaltener Zeitstempel.
   TTimestamp              maxTime;          ///< Grter in jeder der Sequenzen enthaltener Zeitstempel.

   /*!
    * \brief   Die bei einer getriggerten Messung zum Triggerzeitpunkt (t=0) eingestellte Triggerschwelle.
    *          Bei ungetriggerter Messung ist das \ref TOptionalValue::eOption "Gltigkeitsflag" auf 
    *          \ref eOptIgnore zu setzen; der \ref TOptionalValue::value "Nutzwert" darf in diesem Fall
    *          von der Anwendung \em nicht ausgewertet werden.
    */
   TOptionalValue    usedTriggerLevel;
   /*!
    * \brief   Die zu einer Messwert-Sequenz gehrige fortlaufende Nummer. Die Zhlung wird durch Start der die 
    *          \ref xm::Channel::start "Messwertbertragung auslsenden Operation" 
    *          auf Null gesetzt und bei jeder folgenden Sequenz um 1 (in Worten: eins) inkrementiert. Falls
    *          bei der Operation \ref xm::Channel::start "start" ber Angabe von
    *          "hookedChannels" simultan gestartet werden, erhalten innerhalb dieses Messvorganges zeitgleich
    *          gemessene Blcke die jeweils gleiche fortlaufende Nummer.
    *
    *          Der Treiber muss sicherstellen, dass die anfallenden daten in der richtigen Reihenfolge 
    *          bertragen werden, d.h. die erste Sequenz mit der lauflaufenden Nummer N+1 darf erst 
    *          bertragen werden, nachdem auch die letzte Sequenz mit der laufenden Nummer N bertragen wurde.
    *          Wird diese Regel verletzt, ist dies anwendungsseitiug als Datenverlust zu interpretieren; darauf kann
    *          durch entsprechende Hinweise oder Aussetzen der Darstellung fr den/die betreffenden Kanal/ Kanle
    *          reagiert werden. Versptet gesendete Pakete knnen durch die Anwendung optional jedoch auch nachtrglich
    *          noch verarbeitet werden, ohne dass jedoch eine Verpflichtung dazu besteht (d.h. nach Empfangen der
    *          N+1-ten Sequenz drfen danach eintreffende Sequenzen mit der Nummer N ignoriert werden).
    */
   A_UINT32          ulSequenceNumber;

} TMeasurementCycle;


/*!
 * \brief   Die verschiedenen Aktivierungsstatus zu einem Kanal.
 *
 * \note    Detaillierungen der Grnde, warum ggf. ein bestimmter Aktivierungsstatus an einem Kanal
 *          anliegt, sind ggf. ber den "InspectionStatus" zu bertragen.
 */
typedef enum TeActivationStatus
{
   eRunning,         ///< Der Kanal kann Daten liefern.
   eFrozen,          ///< Der Bediener hat vom Gert aus angefordert, die Darstellung der Messwerte fr diesen Kanal einzufrieren.
   eNotReady,        ///< Das Gert hat erkannt, dass der auf dem betreffenden Kanal angeschlosse Tastkopf (Sensor) noch nicht zur Messung bereit ist.
   eNotConnected     ///< Das Gert hat erkannt, dass fr den betreffenden Kanal kein (geeigneter) Tastkopf (Sensor) angeschlossen ist.
   
} TeActivationStatus;


/*!
 * \brief   Die verschiedenen Filtertypen zu einem Kanal.
 */
typedef enum TeFilterType
{
   eFltOff,          ///< Filter ist ausgeschaltet.
   eFltLowPass,      ///< Tiefpassfilter, d.h. nur die \em unterhalb einer Grenzfrequenz liegenden Anteile passieren den Filter nahezu ungeschwcht.
   eFltHighPass,     ///< Hochpassfilter, d.h. nur die \em oberhalb einer Grenzfrequenz liegenden Anteile passieren den Filter nahezu ungeschwcht.
   eFltBandPass,     ///< Bandpassfilter, d.h. nur die \em innerhalb eines Frequenzbandes liegenden Anteile passieren den Filter nahezu ungeschwcht.
   eFltBandStop      ///< Kerbfilter, d.h. nur die \em auerhalb eines Frequenzbandes liegenden Anteile passieren den Filter nahezu ungeschwcht.
   
} TeFilterType;


/*!
 * \brief   Frequenzband.
 */
typedef struct TPassband
{
   TValue   lowerLimitFrq; ///< Untere Grenzfrequenz, bei der eine Dmpfung des Signals um 3 dB eintritt.
   TValue   upperLimitFrq; ///< Obere Grenzfrequenz, bei der eine Dmpfung des Signals um 3 dB eintritt.
   
} TPassband;


/*!
 * \brief   Datentyp zur Festlegung einer Filtermethode und -parametern.
 */
typedef union TuFilter switch (TeFilterType)
{
   case eFltLowPass:    TValue      lowerLimitFrq; ///< Grenzfrequenz fr Tiefpass.
   case eFltHighPass:   TValue      upperLimitFrq; ///< Grenzfrequenz fr Hochpass.
   case eFltBandPass:   TPassband   passband;      ///< Frequenzband fr Bandpass.
   case eFltBandStop:   TPassband   stopband;      ///< Frequenzband fr Kerbfilter.
 /*case eFltOff:        datatype intentionally omitted - not required here */
   
} TuFilter;


/*!
 * \brief   Mgliche Zustnde eines Kanals.
 */
typedef enum TeChannelState
{
   eBasic,                 ///< Grundzustand.
   ePreparedForDataStream, ///< Kanal ist fr eine kontinuierliche Messung eingerichtet.
   ePreparedForSingleData, ///< Kanal ist fr eine Einzelwerterfassung eingerichtet.
   eMeasuring              ///< Kanal fhrt eine kontinuierliche Messung durch.
   
} TeChannelState;


/*!
 * \brief   Ergebnis einer Einzelmessung.
 */
typedef union TuResultValue switch(TePhysicalQuantity)
{
   case eDiodeCheck: TeDiodeResult  diodeResult;   ///< Ergebnis fr eine Diodenprfung.
   default:          TValue         resultValue;   ///< Ergebnis fr alle anderen Messungen (nicht Diodenprfung).

} TuResultValue;


/*!
 * \brief   Information bzgl. eines gekoppelten Kanals und seiner Parametrierung fr die Vorbereitung kontinuierlicher Messungen.
 *
 */
typedef struct THookedChannelInfo
{
   TChannelName      channelName;   ///< Name des zu koppelnden Kanals.
   TOptionalValue    expectedValue; ///< Erwartetetr Wert fr den gekoppelten Kanal.
   
} THookedChannelInfo;


/*!
 * \brief   Sequenz von mehreren gekoppelten Kanlen und ihrer Parametrierinformation.
 */
typedef sequence<THookedChannelInfo> TsqHookedChannelInfo;


/*!
 * \brief   Struktur fr die Zusammenfassung der aktuellen Werte von Auflsung und Messbereich.
 *
 * \note    Auflsung und Messbereich stehen in direkter Abhngigkeit zueinander
 *          und werden daher in diesme gemeinsamen Datentyp gehalten.
 */
typedef struct TRangeAndResolution
{
   TRange      range;         ///< Der aktuell eingestellte Messbereich.
   TValue      resolution;    ///< Die im eingestellten Messbereich erreichbare Auflsung.
   
} TRangeAndResolution;

typedef struct TVersion
{
   A_UINT8        master;
   A_UINT8        major;
   A_UINT8        minor;
   A_UINT8        build;

} TVersion;
