/*!
 * \file    hvmt-type.h
 *
 * \brief   Spezifische Datentypen fuer externe Hochvolt-Messgeraete
 *
 *          Die Datentypen, die in dieser Datei deklariert werden,
 *          sind spezifisch fuer den Zweck "Hochvolt-Messtechnik"
 *
 * \version 1.8.2.0
 * \date    17.10.2017
 * \author  rd electronic GmbH (rde)
 *
 * \hist    \li 1.8.2.0		31.01.2012	rde		neuer Fehlercode eErrNotContacted (Spannungsmessung, Kontaktpruefung)
 *          \li 1.8.1.0		31.01.2012	rde		neuer Fehlercode eErrNotSupported
 * 		    \li 1.8.0.0		27.01.2012	rde		Diodenwerte korrigiert eOpenCircuit <-> eShortCircuit
 * 		    \li 1.7.0.0		20.01.2012	rde		removing of errors : RESISTANCE_TO_HIGH
 *          \li 1.6.0.0		05.01.2012	rde		neu : TeDiodeDescription::eUnspecified
 *												neue Fehler : eErSoftwareError , eErrNotConnected , eErrValueUnderflow , eErrValueOverflow
 *												entfernte Fehler : 	eErSelftestMissed , eErDeviceSafetyCheckMissed
 *																	eErDischargingMissed , eErConnectionBroken
 * 		    \li 1.5.0.0		21.12.2011	rde		Ergaenzung Fehlercode eErStateTimeout(4)
 *          \li 1.4.0.0		12.12.2011	rde		Ergaenzung (eErUnknownError)
 *												Reihenfolge geaendert eMmResistanceMeasurement, eMmDiodeMeasurement, eMmPotentialEqualizationMeasurement
 *												Zustaende ueberarbeitet eMsWaitForButtonPressXXX (eMsWaitForButtonPress3 entfernt)
 *												Fehlercodes ueberarbeitet (Draft-Version)
 * 		    \li 1.1.0.0		17.11.2011	rde		Ergaenzung fehlender Parameter in getFMVersionavaiable, getDeviceIdentification
 * 			\li	1.0.0.0		11.11.2011	rde		Erstellung
 */


/*!
 * \brief   Moegliche Ergebnisse einer Diodenpruefung.
 */
typedef enum TeDiodeResult
{
   eUnspecified     = 0x00,  ///< Das Ergebnis liegt noch nicht vor.
   eForwardBias		= 0xA1,  ///< Durchlassrichtung, d.h. die technische Stromrichtung verlaeuft von der Anode zur Kathode.
   eReverseBias		= 0xA2,  ///< Sperrrichtung, d.h. die technische Stromrichtung verlaeuft von der Kathode zur Anode.
   eOpenCircuit		= 0xA3,  ///< Unterbrechung (Diode in beiden Richtungen nichtleitend).
   eShortCircuit	= 0xA4   ///< Kurzschluss (Diode in beiden Richtungen leitend).
   
} TeDiodeResult;


/*!
 * \brief   Aktueller Modus des Geraetes.
 */
/*!
 * \brief   Der Modus kann ueber die entsprechende prepare-Operationen 
 *          und via \ref HVMT.Device.finishCurrentMode veraendert werden.
 *
 * \image   html  MeasurementMode.png "Diagramm der Messmodi des Messgeraetes" width=15cm
 * \image   latex MeasurementMode.png "Diagramm der Messmodi des Messgeraetes" width=15cm
 */
typedef enum TeMeasurementMode
{
	/** \brief 	Das Gert ist im Modus "Grundzustand".\n
	 * 			Der Modus wird nach erfolgreichen Aufrufen von \ref HVMT.Device.finishCurrentMode und
	 *			\ref HVMT.Device.connect erreicht.\n
     */
	eMmBasic,
   
	/** \brief 	Das Gert ist im Modus "Spannungsmessung".\n
	 * 			Der Modus wird nach erfolgreichem Aufruf von \ref HVMT.Device.prepareVoltageMeasurement erreicht.\n
	 */
	eMmVoltageMeasurement,
	
	/** \brief 	Das Gert ist im Modus "Hochvolt-Isolationsmessung".\n
	 * 			Der Modus wird nach erfolgreichem Aufruf von \ref HVMT.Device.prepareHVIsolationMeasurement erreicht.\n
	 */
	eMmIsolationMeasurement,
	
	/** \brief 	Das Gert ist im Modus "Hochvolt-Isolationsmessung nach SAE J1766".\n
	 * 			Der Modus wird nach erfolgreichem Aufruf von \ref HVMT.Device.prepareHVIsolationSAEJ1766Measurement erreicht.\n
	 */
	eMmIsolationSAEJ1766Measurement,
	
	/** \brief 	Das Gert ist im Modus "Selbsttest".\n
	 * 			Der Modus wird nach erfolgreichem Aufruf von \ref HVMT.Device.prepareSelftest erreicht.\n
	 */
	eMmSelftest,
	
	/** \brief 	Das Gert ist im Modus "Gerte-Sicherheitstest".\n
	 * 			Der Modus wird nach erfolgreichem Aufruf von \ref HVMT.Device.prepareDeviceSafetyCheck erreicht.\n
	 */
	eMmDeviceSafetyCheck,
	
	/** \brief 	Das Gert ist im Modus "Widerstandsmessung".\n
	 * 			Der Modus wird nach erfolgreichem Aufruf von \ref HVMT.Device.prepareDeviceSafetyCheck erreicht.\n
	 */
	eMmResistanceMeasurement,
	
	/** \brief 	Das Gert ist im Modus "Diodenmessung".\n
	 * 			Der Modus wird nach erfolgreichem Aufruf von \ref HVMT.Device.prepareDiodeMeasurement erreicht.\n
	 */
	eMmDiodeMeasurement,
	
	/** \brief 	Das Gert ist im Modus "Potentialausgleichsmessung.".\n
	 * 			Der Modus wird nach erfolgreichem Aufruf von \ref HVMT.Device.preparePotentialEqualizationMeasurement erreicht.\n
	 */
	eMmPotentialEqualizationMeasurement
   
} TeMeasurementMode;


/*!
 * \brief   Moegliche Zustaende des Geraetes.
 */
typedef enum TeError
{
		eErNoError							=  0,	///< Es liegen keine Fehler vor.
		
        eErWrongMode						=  1,   ///< Die Funktionalitt ist im aktuellen Modus nicht verfgbar.
		
        eErWrongState						=  2,   ///< Die Funktionalitt ist im aktuellen Zustand nicht verfgbar.
		
        eErAbort							=  3,   ///< Der Programmablauf wurde unterbrochen.
		
        eErStateTimeout                     =  4,   ///< Der Zustand wurde nicht innerhalb einer bestimmten Zeit gendert.
		
        eErTimeout							=  5,   ///< Timeout fr die Operation ist abgelaufen.
		
        eErButtonReleasedTooEarly			=  6,   ///< Die Messung wurde abgebrochen, weil der Taster zu frh losgelassen wurde.
		
        eErSelftestFailed					=  7, 	///< Selbsttest fehlgeschlagen.
		
        eErDeviceSafetyCheckFailed			= 10,   ///< Gertesicherheitstest fehlergeschalgen.
		
        eErCalibrationFailed				= 11, 	///< Fehler bei der Kalibrierung.
		
        eErCalibrationMissed				= 12, 	///< Die Kalibrierung wurde nicht ausgefhrt.
		
        eErDischargingFailed				= 13,   ///< Die Prfspannung wurde nicht abgebaut.
		
        eErCommunicationCorrupted			= 16,   ///< Die Verbindung zum Gert ist fehlerhaft.
		
        eErValueNotConverged				= 17,   ///< Der Messwert konvergiert nicht.
		
        eErVersionConflict					= 19,   ///< Eine bentigte Komponente ist nicht kompatibel mit den Versionsanforderungen des Gertetreibers.
		
        eErDeviceError						= 20,   ///< Das HV Messgert lieferte einen internen Gertefehler.

		eErOverCurrent						= 21,	///< Ueberstrom : Der Bereich von ( Wert <= 1,5mA ) wurde ueberschritten.
		
		eErOverVoltage						= 22,	///< Ueberspannung : Der Bereich von ( -1,2kV <= Wert <= 1,2kV ) wurde unter-/ueberschritten.
		
		eErVoltageNotReached				= 23,	///< Hochspannung konnte nicht bis zum Zielwert aufgebaut werden.
		
		eErVoltageCheck						= 24,	///< Die HV Ueberpruefung zu Beginn jeder Spannungsmessung ist fehlgeschlagen.
		
		eErMeasurementCurrentNotSupported 	= 25,	///< Der Pruefstrom wird nicht unterstuetzt.
		
		eErTestVoltageNotSupported 			= 26,  	///< Die ausgewhlte/verwendete Prfspannung wird vom Gert nicht untersttzt.
		
		eErInternalResistanceNotSupported 	= 27,  	///< Der interne Widerstand der Spannungsmessung wird nicht unterstuetzt.
		
		eErCurrentNotReached				= 28,	///< Der Pruefstrom wurde nicht erreicht.
		
		eErPotentialVoltage					= 29,	///< Das Fahrzeug ist nicht Spannungsfrei.
		
		eErGdiSoftwareNotReached			= 30,	///< Das ASAM GDI System kann eine notwendige Bibliothek nicht finden.
		
		eErSoftwareError					= 31,	///< Es ist ein Softwarefehler aufgetreten.
		
		eErrNotConnected					= 32,	///< Es wurde noch keine Gerteverbindung aufgebaut.
		
		eErrValueUnderflow					= 33,	///< Der Messwerte liegt unterhalb des Messbereichs.
		
		eErrValueOverflow					= 34,	///< Der Messwerte liegt oberhalb des Messbereichs.
		
		eErrNotSupported					= 35,	///< Fr die Messung bentigen Sie mindestens das Hochvolt-Messmodul VAS6558A.
		
		eErrNotContacted                    = 36,	///< Mindestens eine Prfspitze hat keinen Kontakt zur Messstelle.
		
		eErUnknownError                     = 0x7FFF///< Es ist ein nicht spezifizierter Fehler aufgetreten.
		
} TeError;


/*!
 * \brief   Mgliche Messzustaende des Geraetes.
 *
 * \note	Die Stati werden in verschiedenen Messmodi genutzt.
 *			Damit haben die Stati erst in Verbindung mit dem \ref TeMeasurementMode "Messmodus" Aussagekraft.
 */
typedef enum TeMeasurementState
{
	eMsBasic							= 0xFFFE		, ///< Wenn Mode = NonInitialised
	//eMsError							= 0xFFFF		, ///< Nur in den ODIS-APIs
	eMsWaitForButtonPress1 				= 0				, ///< Warte auf Tastendruck vor Messung
	eMsMeasuring 						= 1				, ///< Messung
	eMsIdle 							= 2				, ///< Warte auf Start (Messung) oder Ende
	eMsFinished 						= 3				, ///< Messung wurde beendet
	//eMsCharging 						= 4				, ///< Hochspannung wird geladen
	eMsDischarging 						= 5				, ///< Hochspannung wird entladen
	eMsCalibrating 						= 6				, ///< Das Geraet fuehrt eine Kalibrierung durch
	eMsWaitForButtonPress2 				= 10			, ///< Warte auf Tastendruck vor 1.Teilmessung SAE oder vor Kalibrierung
	eMsMeasuringSAE1 					= 11			, ///< (Messung SAEJ1766 ) 1. Messung
	eMsSAE1Finished 					= 12			, ///< (Messung SAEJ1766 )Teilmessung 1 wurde beendet
	
	eMsMeasuringSAE2 					= 13			, ///< (Messung SAEJ1766 ) 2. Messung
	eMsSAE2Finished 					= 14			, ///< (Messung SAEJ1766 )Teilmessung 2 wurde beendet
	
	//eMsWaitForButtonPress3 			= 15			, ///< Warte auf Tastendruck vor 3.Teilmessung bei SAEJ1766
	eMsMeasuringSAE3 					= 16			, ///< (Messung SAEJ1766 ) 3. Messung
	eMsSAE3Finished 					= 17			, ///< (Messung SAEJ1766 )Teilmessung 3 wurde beendet
	
	eMsMeasuringSAE4 					= 18			  ///< (Messung SAEJ1766 ) 4. Messung
	//eMsSAE4Finished 					= eMsFinished 	  ///< (Messung SAEJ1766 )Teilmessung 4 wurde beendet

} TeMeasurementState;


/*!
 * \brief   Moegliche Zustaende des Tasters an der Messspitze.
 */
typedef enum TeButtonState
{
	eBsReleased							= 0, ///< Pruefspitzen-Taste nicht gedrueckt.
	eBsPressed							= 1  ///< Pruefspitzen-Taste gedrueckt.
	
} TeButtonState;

/*!
 * \brief	Umfassende Zustandsinformationen des Gertes.
 */
typedef struct TDeviceState
{
	TeMeasurementMode	eMeasurementMode;	///< Zuletzt ermittelter Messmodus.
	TeMeasurementState	eMeasurementState;	///< Zuletzt ermittelter Messzustand.
	TeError				eLastError;			///< Zuletzt ermittelter Gerte- oder Messfehler.
	TeButtonState		eButtonState;		///< Zuletzt ermittelter Tasterzustand.
	
} TDeviceState;

/*!
 * \brief	Typ zur Presentation von Versionsinformationen
 *
 * \note	Ein Versions-Objekt hat folgendes Format : Master.Major.Minor.Build
 */
typedef struct TVersion
{
	short	nMaster;	///< Master number
	short	nMajor;		///< Major number
	short	nMinor;		///< Minor number
	short	nBuild;		///< Build number

} TVersion;
